<?php

namespace App\Controllers\Order;

use App\Controllers\BaseController;
use App\Models\Order\OrderModel;
use CodeIgniter\HTTP\ResponseInterface;
header('Access-Control-Allow-Origin: *'); // Permitir solicitudes desde cualquier origen
header('Access-Control-Allow-Methods: GET, POST, OPTIONS, PUT, DELETE');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

class OrderController extends BaseController
{
    protected $orderModel;

    public function __construct()
    {
        // Cargar el modelo en el constructor
        $this->orderModel = new OrderModel();
    }
    
    public function options()
    {
        // Encabezados CORS
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: GET, POST, OPTIONS, PUT, DELETE');
        header('Access-Control-Allow-Headers: Content-Type, Authorization');
        header('Access-Control-Max-Age: 86400'); // Almacenar respuesta en caché por un día

        return $this->response->setJSON(['status' => 'ok'], 200); // Responder con un status OK
    }
    
    // Obtener todas las órdenes
    public function index()
    {
        $orders = $this->orderModel->where('StatusOrder', 'Pendiente')->findAll();

        return $this->response->setJSON($orders); // Retornar las órdenes en formato JSON
    }

    public function allOrders()
    {
        $AllOrders = $this->orderModel->where('StatusOrder', 'Cancelado')->orWhere('StatusOrder', 'Completado')->findAll();

        return $this->response->setJSON($AllOrders); // Retornar las órdenes en formato JSON
    }

    public function completedOrders()
    {
        $completedOrders = $this->orderModel->where('StatusOrder', 'Completado')->findAll();

        return $this->response->setJSON($completedOrders); // Retornar las órdenes en formato JSON
    }

    public function canceledOrders()
    {
        $canceledOrders = $this->orderModel->where('StatusOrder', 'Cancelado')->findAll();

        return $this->response->setJSON($canceledOrders); // Retornar las órdenes en formato JSON
    }

    // Crear una nueva orden
    public function create()
    {
        $data = $this->request->getJSON(true);
        log_message('info', 'Received data: ' . json_encode($data));

        if (empty($data)) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'No data received.'
            ]);
        }

        if ($this->orderModel->insert($data)) {
            // Obtener el ID de la orden recién insertada
            $insertedId = $this->orderModel->insertID();

            return $this->response->setJSON([
                'status' => 'success',
                'message' => 'Order created successfully.',
                'PK_Order' =>  $insertedId
            ]);
        } else {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Failed to create order.'
            ]);
        }
    }

    


    // Actualizar una orden
    public function update($id)
    {
        // Obtener los datos actualizados
        $data = $this->request->getRawInput();

        // Actualizar la orden
        if ($this->orderModel->update($id, $data)) {
            return $this->response->setJSON(['status' => 'success', 'message' => 'Order updated successfully']);
        } else {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Failed to update order']);
        }
    }

    // Actualizar una orden
    public function updateOrder($id)
    {
        // Obtener los datos actualizados desde el cuerpo de la solicitud (JSON)
        $data = $this->request->getJSON();  // Usar getJSON() para manejar JSON

        // Verificar si los datos fueron recibidos correctamente
        if (!$data) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Invalid JSON data']);
        }

        // Intentar actualizar la tabla con el nuevo estado
        if ($this->orderModel->update($id, (array) $data)) {  // Convierte a array si es un objeto
            return $this->response->setJSON(['status' => 'success', 'message' => 'Data updated successfully']);
        } else {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Failed to update data']);
        }
    }


    // Eliminar una orden
    public function delete($id)
    {
        // Eliminar la orden
        if ($this->orderModel->delete($id)) {
            return $this->response->setJSON(['status' => 'success', 'message' => 'Order deleted successfully']);
        } else {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Failed to delete order']);
        }
    }

    // Obtener una orden específica
    public function show($id)
    {
        $order = $this->orderModel->find($id); // Obtener una orden por ID
        if ($order) {
            return $this->response->setJSON($order); // Retornar la orden en formato JSON
        } else {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Order not found']);
        }
    }

    public function showOrderbyTableNumber($NumTable){
        $order = $this->orderModel->where('TableNumber', $NumTable)->where('StatusOrder', 'Pendiente')->findAll(); // Obtiene el primer resultado que coincida
        if ($order) {
            return $this->response->setJSON([
                'status' => '0', // Indica éxito
                'order' => $order // Incluye la orden encontrada
            ]);
        } else {
            return $this->response->setJSON([
                'status' => '1', 
                'message' => 'No Hay Ordenes pendientes'
            ]);
        }
    }

    public function cancelOrder($id)
    {
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: GET, POST, OPTIONS, PUT, DELETE');
        header('Access-Control-Allow-Headers: Content-Type, Authorization');
    
        // Obtener la orden por su ID
        $order = $this->orderModel->find($id);
    
        if ($order) {
            // Obtener la razón de la cancelación desde la solicitud JSON
            $data = $this->request->getJSON();  // Cambiar a getJSON para datos JSON
            $reason = isset($data->reason) ? $data->reason : 'Sin especificar'; // Obtener la razón desde el JSON
    
            // Verificar si la razón está llegando bien
            log_message('info', 'Razón de cancelación: ' . $reason); // Agregar un log para verificar
    
            // Conexión a la base de datos
            $db = \Config\Database::connect();
    
            // Iniciar una transacción para asegurar que ambas operaciones (insertar y eliminar) se realicen
            $db->transStart();
    
            // Insertar la orden en la tabla t_canceled_order
            $builder = $db->table('t_canceled_order');
            $builder->insert([
                'PK_Order_Canceled' => $order['PK_Order'],
                'DateOrder' => $order['DateOrder'],
                'TotalCost' => $order['TotalCost'],
                'NumCustomers' => $order['NumCustomers'],
                'PaymentType' => $order['PaymentType'],
                'Discount' => $order['Discount'],
                'StatusOrder' => 'Cancelled', // Estado predeterminado 'Cancelled'
                'TableNumber' => $order['TableNumber'],
                'CustomerNotes' => $order['CustomerNotes'],
                'Reason' => $reason, // Guardar la razón seleccionada
                'DeletedAt' => date('Y-m-d H:i:s'), // Fecha y hora actuales como DeletedAt
                'CreatedAt' => $order['CreatedAt'],
                'UpdatedAt' => $order['UpdatedAt']
            ]);
    
            // Eliminar la orden de t_order
            $this->orderModel->delete($id);
    
            // Confirmar la transacción
            $db->transComplete();
    
            // Verificar si la transacción fue exitosa
            if ($db->transStatus() === false) {
                return $this->response->setJSON(['status' => 'error', 'message' => 'Error al cancelar la orden'], 500);
            } else {
                return $this->response->setJSON(['status' => 'success', 'message' => 'Orden cancelada exitosamente'], 200);
            }
        } else {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Orden no encontrada'], 404);
        }
    }
    
    
    
}
