<?php

namespace App\Controllers\OrderDetails;

use App\Controllers\BaseController;
use App\Models\OrderDetails\OrderDetailsModel;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\API\ResponseTrait;
header('Access-Control-Allow-Origin: *'); // Permitir solicitudes desde cualquier origen
header('Access-Control-Allow-Methods: GET, POST, OPTIONS, PUT, DELETE');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

class OrderDetailsController extends BaseController
{
    protected $orderDetailsModel;

    public function __construct()
    {
        $this->orderDetailsModel = new OrderDetailsModel();
    }

    public function options()
    {
        // Encabezados CORS
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: GET, POST, OPTIONS, PUT, DELETE');
        header('Access-Control-Allow-Headers: Content-Type, Authorization');
        header('Access-Control-Max-Age: 86400'); // Almacenar respuesta en caché por un día

        return $this->response->setJSON(['status' => 'ok'], 200); // Responder con un status OK
    }

    // Obtener todas las órdenes
    public function index()
    {
        $orderDetailsModel = $this->orderDetailsModel->findAll(); // Obtener todas las órdenes

        return $this->response->setJSON($orderDetailsModel); // Retornar las órdenes en formato JSON
    }

    // Crear una nueva orden
    public function create()
    {
        $data = $this->request->getJSON(true);
        if (empty($data)) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'No data received.'
            ]);
        }

        if ($this->orderDetailsModel->insert($data)) {
            // Obtener el ID de la orden recién insertada
            $insertedId = $this->orderDetailsModel->insertID();

            return $this->response->setJSON([
                'status' => 'success',
                'message' => 'Order created successfully.',
            ]);
        } else {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Failed to create order.'
            ]);
        }
    }

    public function getByFK($fkOrder)
    {
        if (empty($fkOrder)) {
            return $this->response->setJSON([
                'status' => '0',
                'message' => 'FK_Order is required.'
            ], ResponseInterface::HTTP_BAD_REQUEST);
        }

        // Obtener registros individuales por FK_Order
        $result = $this->orderDetailsModel
            ->select('t_details_order.*, t_product.NameProduct')
            ->join('t_product', 't_details_order.FK_Product = t_product.PK_Product', 'inner')
            ->where('t_details_order.FK_Order', $fkOrder)
            ->findAll();

        // Obtener la suma del SubTotal
        $total = $this->orderDetailsModel
            ->selectSum('SubTotal')
            ->where('FK_Order', $fkOrder)
            ->first();

        if ($result) {
            // Añadir el total dentro de cada detalle
            foreach ($result as &$detail) {
                $detail['total'] = $total['SubTotal'];  // Añadimos el total a cada detalle
            }

            // Responder con los datos y el total
            return $this->response->setJSON([
                'status' => '1',
                'data' => $result
            ]);
        } else {
            return $this->response->setJSON([
                'status' => '0',
                'message' => 'No records found.'
            ], ResponseInterface::HTTP_NOT_FOUND);
        }
    }

    // Actualizar una orden
    public function updateNotes($id)
    {
        // Obtener los datos actualizados desde el cuerpo de la solicitud (JSON)
        $data = $this->request->getJSON();  // Usar getJSON() para manejar JSON

        // Verificar si los datos fueron recibidos correctamente
        if (!$data) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Invalid JSON data']);
        }

        // Intentar actualizar la tabla con el nuevo estado
        if ($this->orderDetailsModel->update($id, (array) $data)) {  // Convierte a array si es un objeto
            return $this->response->setJSON(['status' => 'success', 'message' => 'Notes updated successfully']);
        } else {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Failed to update notes']);
        }
    }
    
    // Eliminar detalles
    public function delete($id)
    {
        // Eliminar detalles
        if ($this->orderDetailsModel->delete($id)) {
            return $this->response->setJSON(['status' => 'success', 'message' => 'Details deleted successfully']);
        } else {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Failed to delete details']);
        }
    }

    public function getAllPendingOrders()
    {
        // Obtener los registros con FK_Order, TableNumber, NameCategory, NameProduct, IsServed, UpdatedAt, Notes, Img y PK_Details
        $result = $this->orderDetailsModel
            ->select("t_details_order.PK_Details, 
                    t_details_order.FK_Order, 
                    t_order.TableNumber, 
                    t_category_product.NameCategory, 
                    t_product.NameProduct, 
                    t_product.Img, 
                    t_details_order.IsServed, 
                    DATE_FORMAT(t_details_order.UpdatedAt, '%H:%i') AS UpdatedAt, 
                    t_details_order.Notes")
            ->join('t_product', 't_details_order.FK_Product = t_product.PK_Product', 'inner')
            ->join('t_category_product', 't_product.FK_Category = t_category_product.PK_Category', 'inner')
            ->join('t_order', 't_details_order.FK_Order = t_order.PK_Order', 'inner')
            ->where('t_order.StatusOrder', 'Pendiente')
            ->where('t_details_order.IsServed', 0)
            ->orderBy('t_details_order.FK_Order, t_category_product.NameCategory, t_details_order.UpdatedAt', 'DESC')
            ->findAll();

        // Organizar los datos en un formato estructurado
        $orders = [];
        foreach ($result as $row) {
            $orderId = $row['FK_Order'];
            $category = $row['NameCategory'];

            // Convertir el blob de la imagen a base64
            $imageBase64 = $row['Img'] ? base64_encode($row['Img']) : null;

            // Si la orden aún no está en el array, se inicializa
            if (!isset($orders[$orderId])) {
                $orders[$orderId] = [];
            }

            // Si la categoría aún no está en la orden, se inicializa
            if (!isset($orders[$orderId][$category])) {
                $orders[$orderId][$category] = [];
            }

            // Agregar cada producto como un registro independiente con su IsServed, UpdatedAt, Notes, Img en base64, TableNumber y PK_Details
            $orders[$orderId][$category][] = [
                'PK_Details' => $row['PK_Details'],  // Agregar PK_Details
                'NameProduct' => $row['NameProduct'],
                'IsServed' => $row['IsServed'],
                'UpdatedAt' => $row['UpdatedAt'],
                'Notes' => $row['Notes'],
                'Img' => $imageBase64 ? "data:image/png;base64," . $imageBase64 : null, // Agregar prefijo de base64
                'TableNumber' => $row['TableNumber'] // Número de mesa dentro del producto
            ];
        }

        // Responder con los datos estructurados
        return $this->response->setJSON([
            'status' => '1',
            'data' => $orders
        ]);
    }
}