<?php

namespace App\Controllers\Products;

use App\Controllers\BaseController;
use App\Models\Products\ProductsModel;
use CodeIgniter\HTTP\ResponseInterface;
header('Access-Control-Allow-Origin: *'); // Permitir solicitudes desde cualquier origen
header('Access-Control-Allow-Methods: GET, POST, OPTIONS, PUT, DELETE');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

class ProductsController extends BaseController
{
    protected $productsModel;

    public function __construct()
    {
        // Cargar el modelo en el constructor
        $this->productsModel = new ProductsModel();
    }
    
    public function options()
    {
        // Encabezados CORS
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: GET, POST, OPTIONS, PUT, DELETE');
        header('Access-Control-Allow-Headers: Content-Type, Authorization');
        header('Access-Control-Max-Age: 86400'); // Almacenar respuesta en caché por un día

        return $this->response->setJSON(['status' => 'ok'], 200); // Responder con un status OK
    }
    
    // Obtener todas las órdenes
    public function index()
    {
        $products = $this->productsModel->findAll();
    
        // Codificar la columna binaria en base64 sin prefijo
        $encodedProducts = array_map(function($product) {
            if (isset($product['Img']) && !empty($product['Img'])) {
                $product['Img'] = base64_encode($product['Img']);
                $product['ImgType'] = 'image/png'; // Cambia esto según el tipo real
            } else {
                $product['Img'] = null;
                $product['ImgType'] = null;
            }
            return $product;
        }, $products);
        
    
        return $this->response->setJSON($encodedProducts);
    }
    

    public function showProductData($id)
    {
        $product = $this->productsModel->find($id); // Obtener un producto por ID
        if ($product) {
            // Verificar si el campo 'Img' existe y no está vacío
            if (!empty($product['Img'])) {
                // Convertir la imagen BLOB a Base64
                $product['Img'] = base64_encode($product['Img']);
            }

            return $this->response->setJSON($product); // Retornar el producto con la imagen en formato Base64
        } else {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Product not found']);
        }
    }


    // Actualizar una orden
    public function updateTableStatus($id)
    {
        // Obtener los datos actualizados desde el cuerpo de la solicitud (JSON)
        $data = $this->request->getJSON();  // Usar getJSON() para manejar JSON

        // Verificar si los datos fueron recibidos correctamente
        if (!$data) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Invalid JSON data']);
        }

        // Intentar actualizar la tabla con el nuevo estado
        if ($this->productsModel->update($id, (array) $data)) {  // Convierte a array si es un objeto
            return $this->response->setJSON(['status' => 'success', 'message' => 'Table updated successfully']);
        } else {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Failed to update table']);
        }
    }

    public function create()
    {
        $data = $this->request->getJSON();
    
        // Verificar si los datos fueron enviados correctamente
        if (!$data) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Invalid JSON data'
            ], ResponseInterface::HTTP_BAD_REQUEST);
        }
    
        // Validar los datos del producto
        $validation = \Config\Services::validation();
        $validation->setRules([
            'NameProduct'       => 'required|string|max_length[255]',
            'FK_Category'       => 'required|integer',
            'CostProduct'       => 'required|decimal',
            'DescriptionProduct'=> 'permit_empty|string',
            'Img'               => 'permit_empty|string', // Validar Base64 opcional
            'IsAvailable'       => 'required|integer|in_list[0,1]',
            'CreatedAt'         => 'required|valid_date',
            'UpdatedAt'         => 'permit_empty|valid_date',
        ]);
    
        if (!$validation->run((array)$data)) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validation->getErrors()
            ], ResponseInterface::HTTP_UNPROCESSABLE_ENTITY);
        }
    
        // Procesar imagen en Base64 si está presente
        if (isset($data->Img) && !empty($data->Img)) {
            try {
                // Decodificar Base64
                $decodedImage = base64_decode($data->Img, true);
    
                // Verificar si la decodificación fue exitosa
                if ($decodedImage === false) {
                    return $this->response->setJSON([
                        'status' => 'error',
                        'message' => 'Invalid Base64 image data'
                    ], ResponseInterface::HTTP_BAD_REQUEST);
                }
    
                // Asignar la imagen decodificada al objeto de datos
                $data->Img = $decodedImage;
            } catch (\Exception $e) {
                return $this->response->setJSON([
                    'status' => 'error',
                    'message' => 'Error processing image: ' . $e->getMessage()
                ], ResponseInterface::HTTP_INTERNAL_SERVER_ERROR);
            }
        } else {
            $data->Img = null; // Si no hay imagen, asignar null
        }
    
        // Insertar los datos en la base de datos
        $inserted = $this->productsModel->insert((array)$data);
    
        if ($inserted) {
            return $this->response->setJSON([
                'status' => 'success',
                'message' => 'Product created successfully',
                'product_id' => $inserted
            ], ResponseInterface::HTTP_CREATED);
        } else {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Failed to create product'
            ], ResponseInterface::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
    // Obtener productos por categoría
    public function getProductsByCategory($categoryId)
    {
        if (!is_numeric($categoryId)) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Invalid category ID'], ResponseInterface::HTTP_BAD_REQUEST);
        }

        // Obtener productos
        $products = $this->productsModel->where('FK_Category', $categoryId)->findAll();

        if (empty($products)) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'No products found'], ResponseInterface::HTTP_NOT_FOUND);
        }

        // Codificar resultados para evitar problemas de caracteres
        $products = array_map(function ($product) {
            if (!empty($product['Img'])) {
                // Asegúrate de que `Img` es un blob válido y codifícalo en Base64
                $product['Img'] = base64_encode($product['Img']);
                // Agregar el tipo MIME de la imagen (ajústalo según el tipo real de tus imágenes)
                $product['ImgType'] = 'image/png'; // Cambia esto si las imágenes son JPEG u otro formato
            }
            return array_map('utf8_encode', $product);
        }, $products);

        return $this->response->setJSON(['status' => 'success', 'data' => $products], ResponseInterface::HTTP_OK);
    }

    public function updateProduct($id)
{
    $data = $this->request->getJSON(true);

    if ($this->productsModel->update($id, $data)) {
        return $this->response->setJSON(['status' => 'success', 'message' => 'Producto actualizado correctamente.']);
    } else {
        return $this->response->setJSON(['status' => 'error', 'message' => 'No se pudo actualizar el producto.'], ResponseInterface::HTTP_BAD_REQUEST);
    }
}
public function updateAvailability()
{
    $data = $this->request->getJSON(true);

    if (!isset($data['productId']) || !isset($data['isAvailable'])) {
        return $this->response->setJSON(['status' => 'error', 'message' => 'Missing required fields'], ResponseInterface::HTTP_BAD_REQUEST);
    }

    $updated = $this->productsModel->update($data['productId'], ['IsAvailable' => $data['isAvailable']]);

    if ($updated) {
        return $this->response->setJSON(['status' => 'success', 'message' => 'Availability updated successfully']);
    } else {
        return $this->response->setJSON(['status' => 'error', 'message' => 'Failed to update availability'], ResponseInterface::HTTP_INTERNAL_SERVER_ERROR);
    }
}

public function deleteProduct($id)
{
    if (!is_numeric($id)) {
        return $this->response->setJSON(['status' => 'error', 'message' => 'Invalid product ID'], ResponseInterface::HTTP_BAD_REQUEST);
    }

    $deleted = $this->productsModel->delete($id);

    if ($deleted) {
        return $this->response->setJSON(['status' => 'success', 'message' => 'Producto eliminado correctamente.']);
    } else {
        return $this->response->setJSON(['status' => 'error', 'message' => 'No se pudo eliminar el producto.'], ResponseInterface::HTTP_INTERNAL_SERVER_ERROR);
    }
}


}
